% Evaluates the results produced by the "nn_classification_PhD" function
% 
% PROTOTYPE
% output = evaluate_results_PhD(results,decision_mode,results1);
% 
% USAGE EXAMPLE(S)
% 
%     Example 1:
%       % we randomly generate a few 100-dimensional training and testing  
%       % feature vectors belonging to 20 classes; 
%       % this shows the use of the 'ID' option
%         n=100;
%         train_feature_vectors=[];
%         test_feature_vectors=[];
%         test_ids = [];
%         train_ids = [];
%         for i=1:20
%             for j=1:10
%                 train_feature_vectors=[train_feature_vectors,i+1.5*randn(n,1)];
%                 train_ids = [train_ids,i];
%                 test_feature_vectors=[test_feature_vectors,i+1.5*randn(n,1)];
%                 test_ids = [test_ids,i];
%             end
%         end    
%         results = nn_classification_PhD(train_feature_vectors, train_ids, test_feature_vectors, test_ids, n, 'euc','all');
%         output = evaluate_results_PhD(results,'ID');
% 
% 
%     Example 2:
%       % we randomly generate a few 100-dimensional training and testing  
%       % feature vectors belonging to 20 classes; 
%       % this shows the use of the 'image' option
%         n=100;
%         train_feature_vectors=[];
%         test_feature_vectors=[];
%         test_ids = [];
%         train_ids = [];
%         for i=1:20
%             for j=1:10
%                 train_feature_vectors=[train_feature_vectors,i+1.5*randn(n,1)];
%                 train_ids = [train_ids,i];
%                 test_feature_vectors=[test_feature_vectors,i+1.5*randn(n,1)];
%                 test_ids = [test_ids,i];
%             end
%         end    
%         results = nn_classification_PhD(train_feature_vectors, train_ids, test_feature_vectors, test_ids, n, 'euc','all');
%         output = evaluate_results_PhD(results,'image');
% 
%     Example 3:
%       % we randomly generate a few 100-dimensional training, testing and  
%       % evaluation feature vectors belonging to 20 classes; 
%       % this shows the use of the 'ID' option and two results structures 
%         n=100;
%         train_feature_vectors=[];
%         test_feature_vectors=[];
%         eval_feature_vectors=[];
%         test_ids = [];
%         train_ids = [];
%         eval_ids = [];
%         for i=1:20
%             for j=1:10
%               if(i<10)
%                 train_feature_vectors=[train_feature_vectors,i+1.5*randn(n,1)];
%                 train_ids = [train_ids,i];
%               end
%                 test_feature_vectors=[test_feature_vectors,i+1.5*randn(n,1)];
%                 test_ids = [test_ids,i];
%                 eval_feature_vectors=[eval_feature_vectors,i+1.5*randn(n,1)];
%                 eval_ids = [eval_ids,i];
%             end
%         end    
%         results = nn_classification_PhD(train_feature_vectors, train_ids, test_feature_vectors, test_ids, n, 'euc','sep');
%         results1 = nn_classification_PhD(train_feature_vectors, train_ids, eval_feature_vectors, eval_ids, n, 'euc','sep');
%         output = evaluate_results_PhD(results,'image',results1);
%
%
% GENERAL DESCRIPTION
% The function evaluates the similarity matrix in the input results
% structure that was produced using the "nn_classification_PhD" function
% from the PhD toolbox and produces performance metrics from the similarity
% matrix. These metrics include: the equal error rate, the verification
% rates at 1%, 0.1% and 1 FAR, the minimal half total eror rate, the rank 
% one recognition rate etc. The function also computes ROC and CMC curve 
% data that can be used to plot ROC and CMC curves. 
% 
% If three inout arguments are provided to the function, it also computes
% EPC curve data that can be used to plot exepcted performance curves. If
% you have installed NISTs DETware, the function also computed DET curve
% data.
% 
% The function operates in to distinct modes, where the results are
% generated in two slightly different ways. If the "decision_mode" input
% argument is set "ID" then all target feature vectors belonging to the
% same ID are considered to comprise the IDs template (e.g., if four scores 
% were generated using target feature vectors with the same ID, the median 
% of these scores is used to make a decision regarding the identity). If
% the "decision_mode" input argument is set to "image" then the results are 
% generated as if pairwise comparison were being made. This means that the 
% templates are considered to be comprised of a single feature vector.
%
% To see an example of usage of the function look at the examples above or 
% find an appropriate demo in the demo folder.
%
% 
% REFERENCES
% There are no specific references linked to this function. Some 
% information on the results generated by this function (e.g., ROC, CMC, 
% EPC curves) can be found in: 
% 
% truc V., Paveic, N.: The Complete Gabor-Fisher Classifier for Robust 
% Face Recognition, EURASIP Advances in Signal Processing, vol. 2010, 26
% pages, doi:10.1155/2010/847680, 2010.
%
%
%
% INPUTS:
% results               - a result structure generated through the use of
%                         the "nn_classification_PhD" function (obligaotry);
%                         for more information on this structure please 
%                         type:
%                               help nn_classification_PhD
% decision_mode         - a string defining the decision mode based on
%                         which to produce the results (optional argument);
%                         valid values:
%                                       'ID' (default)| 'image'
% results1              - a result structure generated through the use of
%                         the "nn_classification_PhD" function (optional);
%                         note that this structure needs to be generated
%                         using test data that is different from that used
%                         to produce the result structure that represents
%                         the obligatory argument to this function; if this
%                         structure is omitted, the function simply skips
%                         the EPC curve calculation that requires additional 
%                         matching results
%
% OUTPUTS:
% output                - an output structure with performance metrics
%                         computed from the similarity matrix; the 
%                         structure contains several fields, which are 
%                         dependant on the input arguments and installed 
%                         software (i.e., if DETware is installed or not). 
%                         In its most complete form the structure contains
%                         the following fields:
% 
%  output.ROC_ver_rate      ... a vector of verification rates that
%                               represents y-axis data of the computed ROC 
%                               curve 
%  output.ROC_miss_rate     ... a vector of miss rates (false acceptance 
%                               rates) that represents x-axis data of the
%                               computed ROC curve
%  output.ROC_char_errors   ... a structure of characteristic error rates
%                               computed from the similarity matrix; it 
%                               contains the following fields:
% 
%       .minHTER_er     ... the minimal achievable half total error rate
%       .minHTER_tr     ... the decision threshold that ensures the above
%                           minimal half total error rate
%       .minHTER_frr    ... the false rejection error rate at the minimal
%                           Half total error rate
%       .minHTER_ver    ... the verification rate at the minimal half
%                           total error rate
%       .minHTER_far    ... the false acceptance error rate at the minimal
%                           half total error rate
%       .EER_er         ... the equal error rate
%       .EER_tr         ... the decision threshold that ensures the above 
%                           equal error rate 
%       .EER_frr        ... the false rejection error rate at the equal 
%                           error rate
%       .EER_ver        ... the verification rate at the equal error rate
%       .FRR_01FAR_er   ... the half total error rate at the operating
%                           point where FRR = 0.1*FAR
%       .FRR_01FAR_tr   ... the decision threshold that ensures the above 
%                           half total error rate of FRR = 0.1*FAR
%       .FRR_01FAR_frr  ... the false rejection error rate at the half 
%                           total error rate of FRR = 0.1*FAR
%       .FRR_01FAR_ver  ... the verification rate at the half 
%                           total error rate of FRR = 0.1*FAR
%       .FRR_01FAR_far  ... the false acceptance error rate at the half 
%                           total error rate of FRR = 0.1*FAR
%       .FRR_10FAR_er   ... the half total error rate at the operating
%                           point where FRR = 10*FAR
%       .FRR_10FAR_tr   ... the decision threshold that ensures the above 
%                           half total error rate of FRR = 10*FAR
%       .FRR_10FAR_frr  ... the false rejection error rate at the half 
%                           total error rate of FRR = 10*FAR
%       .FRR_10FAR_ver  ... the verification rate at the half 
%                           total error rate of FRR = 10*FAR
%       .FRR_10FAR_far  ... the false acceptance error rate at the half 
%                           total error rate of FRR = 10*FAR
%       ...
% 
%       more ...
%   
%       For more information on this structure have a look at the 
%       "rates_and_threshs" structure of the "produce_ROC_PhD" function. 
%       The two structures are identical.
% 
%  output.DET_frr_rate      ... a vector of false rejection rates that
%                               represent y-axis data of the computed DET
%                               curve (this field is only available if 
%                               NISTs DETware is installed)
%  output.DET_far_rate      ... a vector of false acceptance rates that
%                               represent x-axis data of the computed DET
%                               curve (this field is only available if 
%                               NISTs DETware is installed)
%  output.CMC_rec_rates     ... a vector of recognition rates that
%                               represent y-axis data of the computed CMC
%                               curve; NOTE!!: output.CMC_rec_rates(1) 
%                               represents the rank one recognition rate 
%  output.CMC_ranks         ... a vector of ranks that corresponds to the 
%                               recognition rates in "output.CMC_rec_rates"; 
%                               the data represents x-axis data of the 
%                               computed CMC curve
%  output.EPC_char_errors   ... a structure of characteristical error
%                               rates on the EPC curve; for a more detailed
%                               description of this structure look at the
%                               "rates_and_threshs1" output structure of
%                               the "produce_EPC_PhD" function; the two
%                               structures are identical
%  output.EPC_alpha         ... a vector of alpha values that represents
%                               x-axis data of the computed EPC curve;
%                               (this field is only available if two 
%                               results structures obtained on test as well 
%                               as evaluation data are provided as input to 
%                               the function)
%  output.EPC_errors        ... a vector of error rates that represents
%                               y-axis data of the computed EPC curve;
%                               (this field is only available if two 
%                               results structures obtained on test as well 
%                               as evaluation data are provided as input to 
%                               the function)
%
% NOTES / COMMENTS
% The function was tested with Matlab ver. 7.9.0.529 (R2009b) and Matlab 
% ver. 7.11.0.584 (R2010b).
% 
% 
% RELATED FUNCTIONS (SEE ALSO)
% nn_classification_PhD
% return_distance_PhD
% 
% 
% ABOUT
% Created:        10.2.2010
% Last Update:    21.12.2011
% Revision:       1.0
% 
%
% WHEN PUBLISHING A PAPER AS A RESULT OF RESEARCH CONDUCTED BY USING THIS CODE
% OR ANY PART OF IT, MAKE A REFERENCE TO THE FOLLOWING PUBLICATIONS:
% 
% truc V., Paveic, N.: The Complete Gabor-Fisher Classifier for Robust 
% Face Recognition, EURASIP Advances in Signal Processing, vol. 2010, 26
% pages, doi:10.1155/2010/847680, 2010.
%
% truc V., Paveic, N.:Gabor-Based Kernel Partial-Least-Squares 
% Discrimination Features for Face Recognition, Informatica (Vilnius), vol.
% 20, no. 1, pp. 115-138, 2009.
% 
% 
% The BibTex entries for the papers are here
% 
% @Article{ACKNOWL1,
%     author = "Vitomir \v{S}truc and Nikola Pave\v{s}i\'{c}",
%     title  = "The Complete Gabor-Fisher Classifier for Robust Face Recognition",
%     journal = "EURASIP Advances in Signal Processing",
%     volume = "2010",
%     pages = "26",
%     year = "2010",
% }
% 
% @Article{ACKNOWL2,
%     author = "Vitomir \v{S}truc and Nikola Pave\v{s}i\'{c}",
%     title  = "Gabor-Based Kernel Partial-Least-Squares Discrimination Features for Face Recognition",
%     journal = "Informatica (Vilnius)",
%     volume = "20",
%     number = "1",
%     pages = "115138",
%     year = "2009",
% }
% 
% Official website:
% If you have down-loaded the toolbox from any other location than the
% official website, plese check the following link to make sure that you
% have the most recent version:
% 
% http://luks.fe.uni-lj.si/sl/osebje/vitomir/face_tools/PhDface/index.html
%
% 
% OTHER TOOLBOXES 
% If you are interested in face recognition you are invited to have a look
% at the INface toolbox as well. It contains implementations of several
% state-of-the-art photometric normalization techniques that can further 
% improve the face recognition performance, especcially in difficult 
% illumination conditions. The toolbox is available from:
% 
% http://luks.fe.uni-lj.si/sl/osebje/vitomir/face_tools/INFace/index.html
% 
%
% Copyright (c) 2011 Vitomir truc
% Faculty of Electrical Engineering,
% University of Ljubljana, Slovenia
% http://luks.fe.uni-lj.si/en/staff/vitomir/index.html
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files, to deal
% in the Software without restriction, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in 
% all copies or substantial portions of the Software.
%
% The Software is provided "as is", without warranty of any kind.
% 
% December 2011

function output = evaluate_results_PhD(results,decision_mode,results1);
%% Init 
output = [];

%% Check inputs
EPC_flag=0;

%check input arguments
if nargin <1
    disp('The function requires at least one input argument!')
    return;
elseif nargin >3
    disp('The function takes no more than three input arguments!')
    return;
elseif nargin == 1
    decision_mode = 'ID';
elseif nargin == 2
    if ischar(decision_mode)~=1
        disp('The "decision_mode" parameter must be a string.')
        return;
    end
    if strcmp(decision_mode,'ID')==1 || strcmp(decision_mode,'image')==1
        %ok
    else
        disp('The "decision mode" parameter is not valid: ID | image.')
        return;
    end
elseif nargin == 3
    EPC_flag=1;
    if ischar(decision_mode)~=1
        disp('The "decision_mode" parameter must be a string.')
        return;
    end
    
    if strcmp(decision_mode,'ID')==1 || strcmp(decision_mode,'image')==1
    else
        disp('The "decision mode" parameter is not valid: ID | image.')
        return;
    end
    disp('Note that two input structures are used solely for producing verification rates at certain thresholds!')
    disp('This will produce verification rates at certain operating points as well as EPC curve data!')
    disp('In the computation the first input structure will be used for threshold calculation and the second for verification rate calculation.')
end

%check for mode
if isfield(results,'mode') ~= 1
    disp('The results structure contains no definition for "mode". Missing results.mode!');
    return;
end


%check which mode was used
if strcmp(results.mode,'sep')==1
   if isfield(results,'dist')~=1
       disp('The specification about the employed distance could not be found. Missing results.dist!')
       return;
   end
   
   if isfield(results,'dim')~=1
       disp('The specification about the employed feature-dimensionality could not be found. Missing results.dim!')
       return;
   end
   
   if isfield(results,'client_dist')~=1
       disp('Could not find the client distances. Missing results.client_dist!')
       return;
   end
   
   if isfield(results,'same_cli_id')~=1
       disp('Could not find the Id info. Missing results.same_cli_id!')
       return;
   end
   
   if isfield(results,'client_horizontal_ids')~=1
       disp('Could not find client ID info. Missing results.client_horizontal_ids!')
       return;
   end
   
   if isfield(results,'client_vertical_ids')~=1
       disp('Could not find the client ID info. Missing results.client_vertical_ids!')
       return;
   end
   
   if isfield(results,'imp_dist')~=1
       disp('Could not find the impostor distances. Missing results.imp_dist!')
       return;
   end
   
   if isfield(results,'same_imp_id')~=1
       disp('Could not find the impostor ID info. Missing results.same_imp_id!')
       return;
   end
   
   if isfield(results,'imp_vertical_ids')~=1
       disp('Could not find the impostor ID info. Missing results.imp_vertical_ids!')
       return;
   end
   
   if isfield(results,'imp_horizontal_ids')~=1
       disp('Could not find the impostor ID info. Missing results.imp_horizontal_ids!')
       return;
   end
    
    
elseif strcmp(results.mode,'all')==1
%     if nargin == 3
%         disp('The second input structure with results can be used only in the separation mode!')
%         return;
%     end
    
    if isfield(results,'dist')~=1
       disp('The specification about the employed distance could not be found. Missing results.dist!')
       return;
   end
   
   if isfield(results,'dim')~=1
       disp('The specification about the employed feature-dimensionality could not be found. Missing results.dim!')
       return;
   end
    
%    if isfield(results,'same_id')~=1
%        disp('Could not find the ID info. Missing results.same_id!')
%        return;
%    end
   
   if isfield(results,'dist')~=1
       disp('Could not find the similarity matrix. Missing results.match_dist!')
       return;
   end
   
   if isfield(results,'horizontal_ids')~=1
       disp('Could not find the ID info. Missing results.horizontal_ids!')
       return;
   end
   
   if isfield(results,'vertical_ids')~=1
       disp('Could not find the ID info. Missing results.vertical_ids!')
       return;
   end
else
    disp('The input "mode" was not recognized as a valid mode!')
    return;
end



%% Generate results - mode-wise

if strcmp(results.mode,'sep')==1
    
    %account for decision mode
    if strcmp(decision_mode,'ID')==1
        
        %compute ROC and DET data
        disp('Computing ROC and DET curves ...')
        
        %get client distances (median distance in each ID)
        unique_cli_id = unique(results.client_horizontal_ids);
        [y,x] = find(results.same_cli_id==1);
        [Y,X] = meshgrid(results.client_vertical_ids,results.client_horizontal_ids);
        Y=X';
        [y_siz,x_siz] = size(results.same_cli_id);
        
        client_coors = [y,x];
        cont = 1;
        for i=1:length(unique_cli_id)
            
            %find i-th ID that corresponds to client coors
            [y,x] = find(unique_cli_id(i)==Y);
            ID_coors = [y,x];
            score_coors = intersect(ID_coors, client_coors,'rows');
            
            %look at size of ID client matches - there can be more than one
            same_test_ID = unique(score_coors(:,1)); %get the unique y coordinates of the same test
            
            for j=1:length(same_test_ID)
                [y,x]=find(same_test_ID(j)==score_coors(:,1));
                scores_c = sub2ind(size(results.same_cli_id),score_coors(y,1),score_coors(y,2));%[score_coors(y,1),score_coors(y,2)];
                d_client(1,cont) = (median(results.client_dist(scores_c)));
                cont=cont+1;
            end
        end
        
        
        %get impostor distances (median distance in each ID)
        [imp_siz,dumm] = size(results.imp_dist);
        [Y,X] = meshgrid(1:imp_siz,results.client_horizontal_ids);
        Y=X'; %this is ok now - I could also adjust (transpose and change?) the above inputs 
        cont = 1;
        d_impostor = zeros(1,length(unique(results.imp_horizontal_ids))*length(results.imp_vertical_ids));
        for i=1:length(unique_cli_id)
            
            %find i-th ID that corresponds to client coors
            [y,x] = find(unique_cli_id(i)==Y);
            ID_coors = [y,x];
            
            %look each test of the same query on the same ID
            test_ID = unique(ID_coors(:,1)); %get the unique y coordinates of the same test
            
            for j=1:length(test_ID)
                [y,x]=find(test_ID(j)==ID_coors(:,1));
                scores_c = sub2ind(size(results.imp_dist),ID_coors(y,1),ID_coors(y,2));%[score_coors(y,1),score_coors(y,2)];
                d_impostor(1,cont) = (median(results.imp_dist(scores_c)));
                cont=cont+1;
            end
        end
        
        
        %compute ROC curve
        [ver_rate, miss_rate, rates_and_threshs] = produce_ROC_PhD(d_client, d_impostor, 5000);
        
        
        %set ROC outputs
        output.ROC_ver_rate = ver_rate;
        output.ROC_miss_rate = miss_rate;
        output.ROC_char_errors = rates_and_threshs;
        
        %report to prompt
        disp('Finished with ROC, starting with DET if available.')
        
        %try to use Nists DET-ware; if not available report in skip
        try
            %compute DET data
            [Pmiss, Pfa] = Compute_DET(-d_client, -d_impostor);
            
            %set DET outputs
            output.DET_frr_rate = Pmiss;
            output.DET_far_rate = Pfa;    
            disp('Finished with DETs!')
        catch
           disp('No DET-ware installed or not in Matlabs path! Make sure that NISTs DET-ware is present on your machine.')
           disp('You can download a gzipped tar-ball of the software from: http://www.itl.nist.gov/iad/mig/tools/DETware_v2.1.targz.htm')
        end
        
        
        %report to prompt
        disp('Computing CMC curves ...')
        
        %finally we compute the cumulative match score curves - CMCs
        [rec_rates, ranks] = produce_CMC_PhD(results);
        
        
        %set CMC outputs
        output.CMC_rec_rates = rec_rates;
        output.CMC_ranks = ranks;
        
        disp('Finished with CMCs!')
        
        
        %compute EPC curves if specified
        if EPC_flag==1
            
            %report to prompt
            disp('Computing EPC curves and independant evaluation error rates ...')
            
            %as for the ROC curves - extract client and impostor scores
            
            %get client distances (median distance in each ID)
            unique_cli_id = unique(results1.client_horizontal_ids);
            [y,x] = find(results1.same_cli_id==1);
            [Y,X] = meshgrid(results1.client_vertical_ids,results1.client_horizontal_ids);
            Y=X';
            [y_siz,x_siz] = size(results1.same_cli_id);

            client_coors = [y,x];
            cont = 1;
            for i=1:length(unique_cli_id)

                %find i-th ID that corresponds to client coors
                [y,x] = find(unique_cli_id(i)==Y);
                ID_coors = [y,x];
                score_coors = intersect(ID_coors, client_coors,'rows');

                %look at size of ID client matches - there can be more than one
                same_test_ID = unique(score_coors(:,1)); %get the unique y coordinates of the same test

                for j=1:length(same_test_ID)
                    [y,x]=find(same_test_ID(j)==score_coors(:,1));
                    scores_c = sub2ind(size(results1.same_cli_id),score_coors(y,1),score_coors(y,2));%[score_coors(y,1),score_coors(y,2)];
                    d_client1(1,cont) = (median(results1.client_dist(scores_c)));
                    cont=cont+1;
                end
            end


            %get impostor distances (median distance in each ID)
            [imp_siz,dumm] = size(results1.imp_dist);
            [Y,X] = meshgrid(1:imp_siz,results1.client_horizontal_ids);
            Y=X'; %this is ok now - I could also adjust (transpose and change?) the above inputs 
            cont = 1;
            d_impostor1 = zeros(1,length(unique(results1.imp_horizontal_ids))*length(results1.imp_vertical_ids));
            for i=1:length(unique_cli_id)

                %find i-th ID that corresponds to client coors
                [y,x] = find(unique_cli_id(i)==Y);
                ID_coors = [y,x];

                %look each test of the same query on the same ID
                test_ID = unique(ID_coors(:,1)); %get the unique y coordinates of the same test

                for j=1:length(test_ID)
                    [y,x]=find(test_ID(j)==ID_coors(:,1));
                    scores_c = sub2ind(size(results1.imp_dist),ID_coors(y,1),ID_coors(y,2));%[score_coors(y,1),score_coors(y,2)];
                    d_impostor1(1,cont) = (median(results1.imp_dist(scores_c)));
                    cont=cont+1;
                end
            end
            
            
            %compute EPC curves
            [alpha,errors,rates_and_threshs1] = produce_EPC_PhD(d_client,d_impostor,d_client1,d_impostor1,rates_and_threshs,100);
            
            %set EPC outputs
            output.EPC_char_errors = rates_and_threshs1;
            output.EPC_alpha = alpha;
            output.EPC_errors = errors;    
        end    
    elseif strcmp(decision_mode,'image')==1
        
        %compute ROC and DET data
        disp('Computing ROC and DET curves ...')
        
        %get client distances
        cont_c = 1;
        for i=1:length(results.client_horizontal_ids)
            for j=1:length(results.client_vertical_ids)
                if results.client_horizontal_ids(1,i)==results.client_vertical_ids(1,j)
                    d_client(1,cont_c) = results.client_dist(j,i);
                    cont_c=cont_c+1;
                end 
            end
        end
        
        %get impostor distances 
        d_impostor = zeros(1,length(results.imp_horizontal_ids)*length(results.imp_vertical_ids));
        cont = 1;
        for i=1:length(results.imp_horizontal_ids)
            for j=1:length(results.imp_vertical_ids)
                d_impostor(1,cont) = results.imp_dist(j,i);
                cont=cont+1;
            end
        end
        
        %compute ROC curve
        [ver_rate, miss_rate, rates_and_threshs] = produce_ROC_PhD(d_client, d_impostor, 5000);
        
        
        %set ROC outputs
        output.ROC_ver_rate = ver_rate;
        output.ROC_miss_rate = miss_rate;
        output.ROC_char_errors = rates_and_threshs;
        
        
        %report to prompt
        disp('Finished with ROC, starting with DET if available.')
        
        %try to use Nists DET-ware; if not available report in skip
        try
            %compute DET data
            [Pmiss, Pfa] = Compute_DET(-d_client, -d_impostor);
            
            %set DET outputs
            output.DET_frr_rate = Pmiss;
            output.DET_far_rate = Pfa;    
            disp('Finished with DETs!')
        catch
           disp('No DET-ware installed or not in Matlabs path! Make sure that NISTs DET-ware is present on your machine.')
           disp('You can download a gzipped tar-ball of the software from: http://www.nist.gov/itl/iad/mig/upload/DETware_v2-1-tar.gz')
        end
        
        
        
        
        %report to prompt
        disp('Computing CMC curves ...')
        
        %finally we compute the cumulative match score curves - CMCs
        [rec_rates, ranks] = produce_CMC_PhD(results);
        
        
        %set CMC outputs
        output.CMC_rec_rates = rec_rates;
        output.CMC_ranks = ranks;
        
        disp('Finished with CMCs!')
        
        
        
        
        %compute EPC curves if specified
        if EPC_flag==1
            
            %report to prompt
            disp('Computing EPC curves and independant evaluation error rates ...')
            
            %as for the ROC curves - extract client and impostor scores
            
             %get client distances
            cont_c = 1;
            for i=1:length(results1.client_horizontal_ids)
                for j=1:length(results1.client_vertical_ids)
                    if results1.client_horizontal_ids(1,i)==results1.client_vertical_ids(1,j)
                        d_client1(1,cont_c) = results1.client_dist(j,i);
                        cont_c=cont_c+1;
                    end 
                end
            end

            %get impostor distances 
            d_impostor1 = zeros(1,length(results1.imp_horizontal_ids)*length(results1.imp_vertical_ids));
            cont = 1;
            for i=1:length(results1.imp_horizontal_ids)
                for j=1:length(results1.imp_vertical_ids)
                    d_impostor1(1,cont) = results1.imp_dist(j,i);
                    cont=cont+1;
                end
            end
            
            
            %compute EPC curves
            [alpha,errors,rates_and_threshs1] = produce_EPC_PhD(d_client,d_impostor,d_client1,d_impostor1,rates_and_threshs,100);
            
            %set EPC outputs
            output.EPC_char_errors = rates_and_threshs1;
            output.EPC_alpha = alpha;
            output.EPC_errors = errors;    
        end    
    else
       disp('The entered decision mode is not supproted!')
       return;        
    end
    
    
    
elseif strcmp(results.mode,'all')==1
    
    %account for the decision mode
    if strcmp(decision_mode,'ID')==1
        
        %compute the number of client and impostor ID-dependant experiments
        %to conduct
        [unique_id, ind, dummy] = unique(results.horizontal_ids);
        client_exp_num = sum(sum(results.same_cli_id(:,ind)));
        imp_exp_num = sum(sum(1-results.same_cli_id(:,ind)));
        
        %initialize client and impostor distance vectors
        d_client = zeros(1,client_exp_num);
        d_impostor = zeros(1,imp_exp_num);
        
        cont_c = 1;
        cont_i = 1;
        for j=1:length(unique_id) 
            [dum,ind] = find(unique_id(j)==results.horizontal_ids);
%             scores_c = sub2ind(size(results.same_cli_id),dum,ind);
            for i=1:length(results.vertical_ids)
               if results.vertical_ids(i)==unique_id(j)
                    d_client(1,cont_c) = mean(results.match_dist(i,ind));
                    cont_c=cont_c+1;
               else
                   d_impostor(1,cont_i) = mean(results.match_dist(i,ind));
                    cont_i=cont_i+1;
               end
           end
        end
            
        
        %compute ROC curve
        [ver_rate, miss_rate, rates_and_threshs] = produce_ROC_PhD(d_client, d_impostor, 5000);
        
        
        %set ROC outputs
        output.ROC_ver_rate = ver_rate;
        output.ROC_miss_rate = miss_rate;
        output.ROC_char_errors = rates_and_threshs;
        
        %report to prompt
        disp('Finished with ROC, starting with DET if available.')
        
        %try to use Nists DET-ware; if not available report in skip
        try
            %compute DET data
            [Pmiss, Pfa] = Compute_DET(-d_client, -d_impostor);
            
            %set DET outputs
            output.DET_frr_rate = Pmiss;
            output.DET_far_rate = Pfa;    
            disp('Finished with DETs!')
        catch
           disp('No DET-ware installed or not in Matlabs path! Make sure that NISTs DET-ware is present on your machine.')
           disp('You can download a gzipped tar-ball of the software from: http://www.itl.nist.gov/iad/mig/tools/DETware_v2.1.targz.htm')
        end
        
        
        %report to prompt
        disp('Computing CMC curves ...')
        
        %finally we compute the cumulative match score curves - CMCs
        [rec_rates, ranks] = produce_CMC_PhD(results);
        
        
        %set CMC outputs
        output.CMC_rec_rates = rec_rates;
        output.CMC_ranks = ranks;
        
        disp('Finished with CMCs!')
        
        
        
        
         %compute EPC curves if specified
        if EPC_flag==1
            
            %report to prompt
            disp('Computing EPC curves and independant evaluation error rates ...')
            
            %as for the ROC curves - extract client and impostor scores
            
            [unique_id, ind, dummy] = unique(results1.horizontal_ids);
            client_exp_num = sum(sum(results1.same_cli_id(:,ind)));
            imp_exp_num = sum(sum(1-results1.same_cli_id(:,ind)));

            %initialize client and impostor distance vectors
            d_client1 = zeros(1,client_exp_num);
            d_impostor1 = zeros(1,imp_exp_num);

            cont_c = 1;
            cont_i = 1;
            for j=1:length(unique_id) 
                [dum,ind] = find(unique_id(j)==results1.horizontal_ids);
    %             scores_c = sub2ind(size(results.same_cli_id),dum,ind);
                for i=1:length(results1.vertical_ids)
                   if results1.vertical_ids(i)==unique_id(j)
                        d_client1(1,cont_c) = median(results1.match_dist(i,ind));
                        cont_c=cont_c+1;
                   else
                       d_impostor1(1,cont_i) = median(results1.match_dist(i,ind));
                        cont_i=cont_i+1;
                   end
               end
            end
            
            
            %compute EPC curves
            [alpha,errors,rates_and_threshs1] = produce_EPC_PhD(d_client,d_impostor,d_client1,d_impostor1,rates_and_threshs,100);
            
            %set EPC outputs
            output.EPC_char_errors = rates_and_threshs1;
            output.EPC_alpha = alpha;
            output.EPC_errors = errors;    
        end 
    elseif strcmp(decision_mode,'image')==1
        
        %compute the number of client and impostor experiments
        %to conduct
        client_exp_num = sum(sum(results.same_cli_id));
        imp_exp_num = sum(sum(1-results.same_cli_id));
        
        %initialize client and impostor distance vectors
        d_client = zeros(1,client_exp_num);
        d_impostor = zeros(1,imp_exp_num);
        
        cont_c = 1;
        cont_i = 1;
        for j=1:length(results.horizontal_ids) 
            for i=1:length(results.vertical_ids)
               if results.vertical_ids(i)==results.horizontal_ids(j)
                    d_client(1,cont_c) = median(results.match_dist(i,j));
                    cont_c=cont_c+1;
               else
                   d_impostor(1,cont_i) = median(results.match_dist(i,j));
                    cont_i=cont_i+1;
               end
           end
        end
        
        %compute ROC curve
        [ver_rate, miss_rate, rates_and_threshs] = produce_ROC_PhD(d_client, d_impostor, 5000);
        
        
        %set ROC outputs
        output.ROC_ver_rate = ver_rate;
        output.ROC_miss_rate = miss_rate;
        output.ROC_char_errors = rates_and_threshs;
        
        
        %report to prompt
        disp('Finished with ROC, starting with DET if available.')
        
        %try to use Nists DET-ware; if not available report in skip
        try
            %compute DET data
            [Pmiss, Pfa] = Compute_DET(-d_client, -d_impostor);
            
            %set DET outputs
            output.DET_frr_rate = Pmiss;
            output.DET_far_rate = Pfa;    
            disp('Finished with DETs!')
        catch
           disp('No DET-ware installed or not in Matlabs path! Make sure that NISTs DET-ware is present on your machine.')
           disp('You can download a gzipped tar-ball of the software from: http://www.itl.nist.gov/iad/mig/tools/DETware_v2.1.targz.htm')
        end
        
        
        
        %report to prompt
        disp('Computing CMC curves ...')
        
        %finally we compute the cumulative match score curves - CMCs
        [rec_rates, ranks] = produce_CMC_PhD(results);
        
        
        %set CMC outputs
        output.CMC_rec_rates = rec_rates;
        output.CMC_ranks = ranks;
        
        disp('Finished with CMCs!')
        
        
        
        
         %compute EPC curves if specified
        if EPC_flag==1
            
            %report to prompt
            disp('Computing EPC curves and independant evaluation error rates ...')
            
            %as for the ROC curves - extract client and impostor scores
            
            %compute the number of client and impostor experiments
            %to conduct
            client_exp_num = sum(sum(results1.same_cli_id));
            imp_exp_num = sum(sum(1-results1.same_cli_id));

            %initialize client and impostor distance vectors
            d_client1 = zeros(1,client_exp_num);
            d_impostor1 = zeros(1,imp_exp_num);

            cont_c = 1;
            cont_i = 1;
            for j=1:length(results1.horizontal_ids) 
                for i=1:length(results1.vertical_ids)
                   if results1.vertical_ids(i)==results1.horizontal_ids(j)
                        d_client1(1,cont_c) = median(results1.match_dist(i,j));
                        cont_c=cont_c+1;
                   else
                       d_impostor1(1,cont_i) = median(results1.match_dist(i,j));
                        cont_i=cont_i+1;
                   end
               end
            end
            
            
            %compute EPC curves
            [alpha,errors,rates_and_threshs1] = produce_EPC_PhD(d_client,d_impostor,d_client1,d_impostor1,rates_and_threshs,100);
            
            %set EPC outputs
            output.EPC_char_errors = rates_and_threshs1;
            output.EPC_alpha = alpha;
            output.EPC_errors = errors;    
        end 
        
        
        
        
        
        
    else
       disp('The entered decision mode is not supproted!')
       return;        
    end  
end













